<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/email_functions.php';
require_once '../includes/whatsapp_functions.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

header('Content-Type: application/json');

function testEmailConfiguration($pdo) {
    try {
        // Get email settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'email_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        $results = [];
        $results['email_enabled'] = ($settings['enable_email'] ?? '0') === '1';
        $results['smtp_host'] = !empty($settings['smtp_host']) ? $settings['smtp_host'] : 'NOT SET';
        $results['smtp_port'] = !empty($settings['smtp_port']) ? $settings['smtp_port'] : 'NOT SET';
        $results['smtp_username'] = !empty($settings['smtp_username']) ? '***@' . substr($settings['smtp_username'], strpos($settings['smtp_username'], '@')) : 'NOT SET';
        $results['smtp_password'] = !empty($settings['smtp_password']) ? 'SET' : 'NOT SET';
        $results['smtp_encryption'] = $settings['smtp_encryption'] ?? 'NOT SET';
        $results['test_email'] = !empty($settings['email_test_address']) ? $settings['email_test_address'] : 'NOT SET';
        
        // Check if all required settings are present
        $required_fields = ['smtp_host', 'smtp_port', 'smtp_username', 'smtp_password'];
        $missing_fields = [];
        foreach ($required_fields as $field) {
            if (empty($settings[$field])) {
                $missing_fields[] = $field;
            }
        }
        
        $results['missing_required_fields'] = $missing_fields;
        $results['configuration_complete'] = empty($missing_fields);
        
        // Test email connection if configuration is complete
        if ($results['configuration_complete'] && $results['email_enabled']) {
            try {
                $test_result = testEmailSettings();
                $results['test_result'] = $test_result;
            } catch (Exception $e) {
                $results['test_result'] = ['success' => false, 'message' => $e->getMessage()];
            }
        }
        
        return $results;
    } catch (Exception $e) {
        return ['error' => 'Failed to check email configuration: ' . $e->getMessage()];
    }
}

function testWhatsAppConfiguration($pdo) {
    try {
        // Get WhatsApp settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        $results = [];
        $results['whatsapp_enabled'] = ($settings['enable_whatsapp'] ?? '0') === '1';
        $results['whatsapp_method'] = $settings['whatsapp_method'] ?? 'web_link';
        $results['test_number'] = !empty($settings['whatsapp_test_number']) ? $settings['whatsapp_test_number'] : 'NOT SET';
        
        switch ($results['whatsapp_method']) {
            case 'api':
                $results['api_key'] = !empty($settings['whatsapp_api_key']) ? 'SET' : 'NOT SET';
                $results['api_url'] = !empty($settings['whatsapp_api_url']) ? $settings['whatsapp_api_url'] : 'NOT SET';
                $results['phone_number_id'] = !empty($settings['whatsapp_phone_number_id']) ? $settings['whatsapp_phone_number_id'] : 'NOT SET';
                $required_fields = ['whatsapp_api_key', 'whatsapp_api_url', 'whatsapp_phone_number_id', 'whatsapp_test_number'];
                break;
            case 'business_app':
                $results['business_phone'] = !empty($settings['whatsapp_business_phone']) ? $settings['whatsapp_business_phone'] : 'NOT SET';
                $required_fields = ['whatsapp_business_phone', 'whatsapp_test_number'];
                break;
            case 'web_link':
            default:
                $required_fields = ['whatsapp_test_number'];
                break;
        }
        
        $missing_fields = [];
        foreach ($required_fields as $field) {
            if (empty($settings[$field])) {
                $missing_fields[] = $field;
            }
        }
        
        $results['missing_required_fields'] = $missing_fields;
        $results['configuration_complete'] = empty($missing_fields);
        
        // Test WhatsApp if configuration is complete
        if ($results['configuration_complete'] && $results['whatsapp_enabled']) {
            try {
                $test_result = testWhatsAppSettings();
                $results['test_result'] = $test_result;
            } catch (Exception $e) {
                $results['test_result'] = ['success' => false, 'message' => $e->getMessage()];
            }
        }
        
        return $results;
    } catch (Exception $e) {
        return ['error' => 'Failed to check WhatsApp configuration: ' . $e->getMessage()];
    }
}

function checkCustomerContactInfo($pdo, $customer_id = null) {
    try {
        $sql = "SELECT id, name, email, phone FROM customers";
        $params = [];
        
        if ($customer_id) {
            $sql .= " WHERE id = ?";
            $params[] = $customer_id;
        } else {
            $sql .= " LIMIT 10"; // Get first 10 customers for testing
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $results = [];
        foreach ($customers as $customer) {
            $results[] = [
                'id' => $customer['id'],
                'name' => $customer['name'],
                'has_email' => !empty($customer['email']),
                'email' => $customer['email'] ?? 'NOT SET',
                'has_phone' => !empty($customer['phone']),
                'phone' => $customer['phone'] ?? 'NOT SET'
            ];
        }
        
        return $results;
    } catch (Exception $e) {
        return ['error' => 'Failed to check customer contact info: ' . $e->getMessage()];
    }
}

// Get test type from request
$test_type = $_GET['test'] ?? 'all';
$customer_id = $_GET['customer_id'] ?? null;

$response = [];

switch ($test_type) {
    case 'email':
        $response['email'] = testEmailConfiguration($pdo);
        break;
    case 'whatsapp':
        $response['whatsapp'] = testWhatsAppConfiguration($pdo);
        break;
    case 'customers':
        $response['customers'] = checkCustomerContactInfo($pdo, $customer_id);
        break;
    case 'all':
    default:
        $response['email'] = testEmailConfiguration($pdo);
        $response['whatsapp'] = testWhatsAppConfiguration($pdo);
        $response['customers'] = checkCustomerContactInfo($pdo);
        break;
}

echo json_encode($response, JSON_PRETTY_PRINT); 